#pragma rtGlobals=2		// Use modern global access method.
#include <Decimation>, menus=0

// mafnum2str (value, precision, with units?) : returns string of value with correct precision and optional units prefix
// checkVal (ctrlname, winname) : returns true if control in window is checked
// extractwavetimes (w) : returns time of wave (in seconds) extracted from wavenote
// changeX2Y (wave, x, y) : goes through wave, changing each occurrence of value X to Y
// asrl (startpt, endpt, height, label) : attach strontium line - labels graph with drug treatment time
// avsrl (startpt, endpt, height, label) : attach vertical strontium line - same as asrl, but vertical
// chopx (w, xval, direction) : deletes point either before or after specified x value, depending on direction = -1 or 1
// mxx (w, startx, endx) // rescales wave so that startx now found at endx
// FindPeakSlope (w, startx, pol) : find peak slope of specified polarity (-1 or 1) in wave, starting at startx
// msx (w, startx, xpos, pol)	 : move peak slope to x position
// MakeGraphAvgSD (graphname, basename, undersample) // calculates average of waves displayed in graph, and generating decimated waves for plotting error bars
// decimaterange (wave, destName, startpt, endpt, factor, renew x?) : decimates wave over given point range by factor, creating x wave
// sb (xlen, ylen) : scale bars, and labels them with values
// sbo (xlen, ylen, xaxis, yaxis) : scale bars other - draws them for specified axes

Menu "Macros"
	"Add Glitch Buttons"
End


function AddToNote (w, mystr)	// adds a string to the wave note, adding semicolons as necessary
	wave w
	string mystr
	string tempstr
	
	if (strlen (mystr) == 0)
		return 0
	endif
	tempstr = note(w)
	note /k w
	if ((strlen (tempstr) > 0) && (cmpstr (tempstr[strlen(tempstr) - 1], ";") != 0))	// do we need to insert a ';'?
		tempstr += ";"
	endif
	Note w, tempstr + mystr
end

function/s mafnum2str (v, prec, units)	// nicely formats a value v, with the number of digits set by prec, and a units prefix if units = 1
	variable v, prec, units
	variable lg, sign, tempval
	string s
	
	sign = 0
	if (v < 0)
		sign = 1
		v *= -1
	endif
	lg = floor (log (v))
	v = round (v / (10 ^ (lg - prec + 1))) * (10 ^ (lg - prec + 1))
	if (units == 1)
		if (lg < 0)
			tempval = prec - (mod (3 + mod (lg, 3), 3) + 1)
		else
			tempval = prec - (mod (lg, 3) + 1)
		endif
		if (tempval < 0)
			tempval = 0
		endif
		sprintf s, "%." + num2str (tempval) + "W1P", v
	else
		s = num2str (v)
	endif
	if (sign == 1)
		s = "-" + s
	endif
	return s
end
function suffix2num (mystr)	// returns the numeric suffix from a string (e.g. "ch0_32" -> 32)
	string mystr
	variable i
	for (i = strlen (mystr) - 1; i >= 0; i -= 1)
		if (((cmpstr (mystr[i], "0") == -1) || (cmpstr (mystr[i], "9") == 1)) && cmpstr (mystr[i], "."))
			break
		endif
	endfor
	return (str2num (mystr[i + 1, inf]))
end
function /t prefix2str (mystr)	// returns the alphabetical prefix from a string (e.g. "ch0_32" -> "ch")
	string mystr
	variable i
	for (i = strlen (mystr) - 1; i >= 0; i -= 1)
		if (((cmpstr (mystr[i], "0") == -1) || (cmpstr (mystr[i], "9") == 1)) && cmpstr (mystr[i], "."))
			break
		endif
	endfor
	return (mystr[0,i])
end
function numberFromList (index, listStr)	// shortcut equivalent of stringFromList, but requires ";"-delimited list
	variable index
	string listStr
	return (str2num (stringfromlist (index, listStr)))
end
function /t RemoveManyByKey (keystr, keyliststr)	// just like RemoveByKey, but keyStr may be a list of many keys to remove
	string keystr, keyliststr
	variable i
	for (i = 0; i < itemsinlist (keystr); i += 1)
		keyliststr = removebykey (stringfromlist (i, keystr), keyliststr)
	endfor
	return (keyliststr)
end

function /t unPadString (s)		// removes padded zeroes from a string (sort of inverse of PadString)
	string s
	variable i
	
	i = strsearch (s, num2char (0), 0)
	if (i == -1)
		return (s)
	else
		return (s[0, i - 1])
	endif
end
function /t wave2list (w, delstr)	// converts a numeric wave to a string list, with delstr as delimiter
	wave w
	string delstr
	string resultstr=""
	variable i
	for (i = 0; i < numpnts (w); i += 1)
		resultstr += num2str (w[i]) + delstr
	endfor
	return resultstr
end
function list2wave (instr, w, delstr)	// converts a list of numbers to a numeric wave, with delstr as delimiter
	string instr, delstr
	wave w
	variable i
//	redimension /n=(itemsinlist (instr, delstr)) w
	for (i = 0; i < itemsinlist (instr, delstr) - 1; i += 1)
		w[i] = str2num (stringfromlist (i, instr, delstr))
	endfor
end

function checkVal (ctrlname, winname)	// returns the current state of a checkbox control from the named window - also useful for popups
	string ctrlname, winname	// use "" for top window
	if (strlen (winname) == 0)
		ControlInfo $ctrlname
	else
		ControlInfo /w=$winname $ctrlname
	endif
	if (v_flag == 0)
		print "Couldn't find \"" + ctrlname + "\" in " + selectstring (strlen (winname) == 0, "window \"" + winname + "\"", "front window")
	endif
	return v_value
end

function /t PopupVal (ctrlname, winname)	// returns the current string displayed by a popup in the named window - use "" for top window
	string ctrlname, winname
	if (strlen (winname) == 0)
		ControlInfo $ctrlname
	else
		ControlInfo /w=$winname $ctrlname
	endif
	if (v_flag == 0)
		print "Couldn't find \"" + ctrlname + "\" in " + selectstring (strlen (winname) == 0, "window \"" + winname + "\"", "front window")
		return ""
	endif
	return s_value
end

function WinRect (mywin, thisrect)
	string mywin
	struct rect &thisrect
end

function extractwavetimes (w)	// returns the time a wave was collected using the wavenote
	wave w
	variable i, hour, min, sec, cpos
	string wnote
	
	wnote = stringbykey ("TIME", note(w))
	cpos = strsearch (wnote, ".", 0)
	hour = str2num (wnote[cpos-2, cpos-1])
	min = str2num (wnote[cpos+1, cpos+2])
	sec = str2num (wnote[cpos+4, cpos+5])
	return (sec + min * 60 + hour * 3600)
end

function changeX2Y (w, xval, yval)	// go through w and changes x's to y's
	wave w
	variable xval, yval
	if (numtype (xval) != 2)
		w = (w == xval ? yval : w)
	else
		w = (numtype (w) == 2 ? yval : w)
	endif
end
function clipNan (w)	// goes through wave and removes points that are Nan
	// of course, this will mess up x-scaling
	wave w
	variable i
	for (i = numpnts (w) - 1; i >= 0; i -= 1)
		if (numtype (w[i]) == 2)
			deletepoints i, 1, w
		endif
	endfor
end

function asrl(startpt, endpt, height, label)		// attach strontium line
	variable startpt, endpt, height
	string label
	
	SetDrawEnv xcoord= bottom,ycoord= left,linethick= 2.00
	DrawLine startpt, height, endpt, height
	if (strlen (label) > 0)
		SetDrawEnv fname="Helvetica",textxjust= 1,textyjust= 0, ycoord=left,xcoord= bottom, fsize=9
		DrawText (startpt + endpt)/2, height, label
	endif
end

function avsrl(startpt, endpt, height, label)	// attach vertical strontium line
	variable startpt, endpt, height
	string label
	
	SetDrawEnv xcoord= bottom,ycoord= left,linethick= 2.00
	DrawLine height, startpt, height, endpt
	if (strlen (label) > 0)
		SetDrawEnv fname="Helvetica",textxjust= 2,textyjust= 1, ycoord=left,xcoord= bottom, fsize=9
		DrawText height, (startpt + endpt)/2, label + " "
	endif
end

function chopx (w, xval, direction)	// deletes points in wave up through xval if direction = -1, or after xval if direction = +1
	wave w
	variable xval, direction

	if (direction == -1)
		if (xval < leftx (w))
			abort ("ChopX error: "+nameofwave(w)+" doesn't contain "+num2str (xval))
		endif
		deletepoints 0, (x2pnt(w, xval)), w
		setscale /P x xval, deltax (w), "", w
	elseif (direction == 1)
		if (xval > rightx (w))
			abort ("ChopX error: "+nameofwave(w)+" doesn't contain "+num2str (xval))
		endif
		deletepoints (x2pnt(w, xval) + 1), numpnts (w), w
	endif
end

function InsertVal (w, val)
	wave w
	variable val
	variable valpos = xwave2pnt (w, val)
	if (w[valpos] == val)
		return 0
	endif
	
	if (w[valpos] < val)
		valpos += 1
	endif
	if (numpnts (w) == 0)
		valpos = 0
	endif
	InsertPoints valpos, 1, w
	w[valpos] = val
	return valpos
end

function mxx (w, startx, endx) // rescales wave to move x to new x
	wave w
	variable	startx, endx	// starting x value, desired x value
	setscale /P x (endx - startx + leftx (w)), deltax(w), "", w
end

function FindPeakSlope (w, startx, pol)		// find peak slope
	wave w
	variable startx, pol
	
	duplicate /o w tempdiff
	differentiate tempdiff
	smooth /b 11, tempdiff
	wavestats /q/r=(startx,) tempdiff
	if (pol == -1)
		return v_minloc
	else
		return v_maxloc
	endif
end

function msx (w, startx, xpos, pol)	// move peak slope to x position
	wave w				// wave to be adjusted
	variable startx		// x position of wave to start looking for peak slope
	variable xpos		// x position to rescale wave to, so that peak slope is there
	variable pol			// is it a plus (1) or a minus (-1) slope?
	variable i, lasty

	duplicate /o w tempdiff
	differentiate tempdiff
	wavestats /q/r=(startx, startx + .1) tempdiff
	if (pol == -1)
		SetScale/P x (xpos - v_minloc + leftx (w)),(deltax(w)),"",w
	else
		SetScale/P x (xpos - v_maxloc + leftx (w)),(deltax(w)),"",w
	endif
end

function MakeGraphAvgSD (graphname, basename, undersample, smartn)	// averages together all waves in a graph, and stores results starting with basename
	string graphname, basename
	variable undersample	// also generates wave with only every few values (e.g. if 10, every 10th value)
	variable smartn	// if smartn = 0, then the n for each x-value is the same as the number of traces
					// if smartn = 1, then the n for each x-value is independent, depending on whether each wave has a value for it.
	variable i, j, n
	string tempstr
	variable startpt, endpt, minx, maxx, mindelta
	
	tempstr = getdatafolder(1)
	setdatafolder root:
	getwindow $graphname, wavelist
	wave /t w_wavelist
	setdatafolder tempstr
	wave wadd = $(w_wavelist[0][1])
	minx = leftx (wadd)
	maxx = pnt2x (wadd, numpnts (wadd) - 1)
	mindelta = deltax (wadd)
	for (i = 1; i < dimsize (w_wavelist, 0); i += 1)
		wave wadd = $(w_wavelist[i][1])
		minx = min (minx, leftx (wadd))
		maxx = max (maxx, pnt2x (wadd, numpnts (wadd) - 1))
		mindelta = min (mindelta, deltax (wadd))
	endfor	
	make /o/n=((maxx - minx) / mindelta + 1) $(basename + "_avg"), $(basename + "_SE"), $(basename + "_n")
	wave wavg = $(basename + "_avg"), wse = $(basename + "_se"), wn = $(basename + "_n")
	setscale /p x, minx, mindelta, wavg, wse, wn
	wavg = 0; wse=0; wn=0
	for (i = 0; i < dimsize (w_wavelist, 0); i += 1)
		wave wadd = $(w_wavelist[i][1])
		startpt = x2pnt (wavg, leftx (wadd))
		endpt = startpt + numpnts (wadd) - 1
		wavg[startpt, endpt] += numtype (wadd(x)) == 2 ? 0 : wadd(x)
		wse[startpt, endpt] += numtype (wadd(x)) == 2 ? 0 : wadd(x) * wadd(x)
		if (smartn)
			wn[startpt, endpt] += numtype (wadd(x)) == 2 ? 0 : 1
		else
			wn += 1
		endif
	endfor
	wse = wn > 1 ? sqrt ((wse - wavg * wavg / wn) / (wn - 1) / wn) : nan
	wavg /= wn
//	duplicate /o wavg $(basename + "_avgp"), $(basename + "_avgm")	// creates two additional waves that are the avg wave +/- the standard error
//	wave wavgp = $(basename + "_avgp")
//	wavgp += wse
//	wave wavgm = $(basename + "_avgm")
//	wavgm -= wse
	if (undersample > 1)
		duplicate /o wavg $(basename + "_avgdec")
		duplicate /o wse $(basename + "_SEdec")
		wave wavgd = $(basename + "_avgdec")
		wave wsed = $(basename + "_SEdec")
		redimension /n=(numpnts (wavgd) / undersample) wavgd, wsed
		SetScale/P x (leftx (wavg)),(deltax(wavg) * undersample),"",wavgd, wsed
		wavgd= wavg[p * undersample]
		wsed = wse[p * undersample]
	endif
end

function MakeCumGraphAvgSD (graphname, basename, undersample)
	string graphname, basename
	variable undersample
	variable i, j, n
	string tempstr
	
	tempstr = getdatafolder(1)
	setdatafolder root:
	getwindow $graphname, wavelist
	wave/t w_wavelist
	wave wadd = $(w_wavelist[0][1])
	duplicate /o wadd $(basename + "_avg")
	duplicate /o wadd $(basename + "_SE")
	duplicate /o wadd $(basename + "_n")
	wave wavg = $(basename + "_avg")
	wave wse = $(basename + "_se")
	wave wn = $(basename + "_n")
	wavg = selectnumber (numtype (wadd) == 2, wadd, 0)
	wse = selectnumber (numtype (wadd) == 2, wadd * wadd, 0)
	wn = selectnumber (numtype (wadd) == 2, 1, 0)
	i = 1
	n = dimsize (w_wavelist, 0)
	do
		wave wadd = $(w_wavelist[i][1])
		wavg += selectnumber (numtype (wadd) == 2, wadd, 0)
		wse += selectnumber (numtype (wadd) == 2, wadd * wadd, 0)
		wn += selectnumber (numtype (wadd) == 2, 1, 0)
		i += 1
	while (i < n)
	wse = sqrt ((wse - wavg * wavg / wn) / (wn - 1) / wn)
	wavg /= wn
	duplicate /o wavg $(basename + "_avgp"), $(basename + "_avgm")
	wave wavgp = $(basename + "_avgp")
	wavgp += wse
	wave wavgm = $(basename + "_avgm")
	wavgm -= wse
	duplicate /o wavg $(basename + "_avgdec")
	duplicate /o wse $(basename + "_SEdec")
	wave wavgd = $(basename + "_avgdec")
	wave wsed = $(basename + "_SEdec")
	redimension /n=(numpnts (wavgd) / undersample) wavgd, wsed
	SetScale/P x (leftx (wavg)),(deltax(wavg) * undersample),"",wavgd, wsed
	wavgd= wavg[p * undersample]
	wsed = wse[p * undersample]
	setdatafolder tempstr
end


function aaa(startpt, endpt, height)	// attach averaging arrow
	variable startpt, endpt, height
	
	SetDrawEnv arrow=3, arrowfat=.5, arrowlen=4, xcoord= bottom,ycoord= left,linethick= 1;DelayUpdate
	DrawLine startpt, height, endpt, height
end

function sb (xlen, ylen)	// scale bars
	variable xlen, ylen
	sbo (xlen, ylen, "bottom", "left")
end
function sbo (xlen, ylen, xaxis, yaxis)	// scale bars "other", to label on other axes
	variable xlen, ylen
	string xaxis, yaxis
	variable xstart,xend, xpos, xjust
	variable ystart,yend
	
	if (strlen (xaxis) < 0 || strlen (yaxis) < 0)
		abort "sbo error: Axes not specified"
	endif
	GetAxis /Q $xaxis; xstart = v_min; xend = v_max
	GetAxis /Q $yaxis; ystart = v_min; yend = v_max
	if (xlen != 0)
		SetDrawEnv xcoord= $xaxis, ycoord = $yaxis
		DrawLine (xstart + xend)/2,(ystart + yend)/2,(xstart + xend)/2 + xlen,(ystart + yend)/2
		SetDrawEnv fname="Helvetica",textxjust= 1,textyjust= 2, xcoord= $xaxis, ycoord=$yaxis, fsize=8
		if (xlen < .1)
			DrawText (xstart + xend + xlen)/2,(ystart + yend)/2, num2str (xlen * 1000) + " ms"
		else
			DrawText (xstart + xend + xlen)/2,(ystart + yend)/2, num2str (xlen) + " s"
		endif
	endif
	if (ylen != 0)
		xpos = (xstart + xend)/2
		xjust=0
		if (ylen < 0)
			xpos += xlen
			ylen *= -1
			xjust = 2
		endif
		SetDrawEnv xcoord= $xaxis, ycoord = $yaxis
		DrawLine xpos,(ystart + yend)/2,xpos,(ystart + yend)/2 + ylen
		SetDrawEnv fname="Helvetica",textxjust= xjust,textyjust= 1, xcoord= $xaxis,ycoord= $yaxis, fsize=8
		if (ylen > 99)
			DrawText xpos,(ystart + yend + ylen)/2," " + num2str (ylen/1000) + " nA "
		elseif (ylen > 20)
			DrawText xpos,(ystart + yend + ylen)/2," " + num2str (ylen) + " pA "
		else
			DrawText xpos,(ystart + yend + ylen)/2," " + num2str (ylen) + "% "
		endif
	endif
end

function findwaveval (w, val)	// find a value within a wave
	wave w
	variable val
	return (findvaluebetter (w, val, 0, 1))
end

function findwaveval2 (w, val, prec)	// finds value in a wave within a certain precision (in percent)
	wave w
	variable val, prec
	variable i
	for (i = 0; i < numpnts (w); i += 1)
		if ((val * (1 - prec) < w[i]) && (w[i] <  val * (1 + prec)))
			return i
		endif
	endfor
	return (-1)	// didn't find it, so fail
end
function findValueBetter (w, val, startp, direction)
	wave w
	variable val, startp, direction
	variable findingNaN = (numtype (val) == 2), i = startp, endp = numpnts (w)
	
	for (i = startp; i >= 0 && i < endp; i += direction)
		if ((findingNaN && numtype (w[i]) == 2) || (!findingNaN && w[i] == val))
			return i
		endif
	endfor
	return -1
end

function find2DValue (w, val1, val2)
	wave w
	variable val1, val2
	variable i, endpt = dimsize (w, 0)
	for (i = 0; i < endpt; i += 1)
		if (w[i][0] == val1)
			if (w[i][1] == val2)
				return i
			endif
		endif
	endfor
	return -1
end
function sort2DBetter (indexname, wname)
	string indexname, wname
	variable i, numrows, numcols, indexIsText=0, dataIsText=0
	string tempstr
	
	if (wavetype ($indexname) == 0)	// first check on types (text waves need to be treated differently
		wave /t wtindex=$indexname
		indexIsText = 1
	else
		wave windex=$indexname
	endif
	if (wavetype ($wname) == 0)
		wave /t wt=$wname
		dataIsText = 1
	else
		wave w = $wname
	endif
	numrows = dimsize (w, 0); numcols = dimsize(w, 1)
	
	for (i = 0; i < numcols; i += 1)	// split columns up into individual waves
		if (dataIsText)
			make /t/o/n=(numrows) $("temp" + num2str (i))
			wave /t wttemp = $("temp" + num2str (i))
			wttemp = wt[p][i]
		else 
			make /o/n=(numrows) $("temp" + num2str (i))		//	/y=(wavetype (w)) ???
			wave wtemp = $("temp" + num2str (i))
			wtemp = w[p][i]
		endif
		if (i == 0)
			tempstr = "temp0"
		else
			tempstr += ",temp" + num2str (i)
		endif
	endfor
	execute ("sort " + selectstring (indexIsText, "", "/a ") + indexname + " " + tempstr)
	for (i = 0; i < numcols; i += 1)	// recombine waves
		if (dataIsText)
			wave /t wttemp = $("temp" + num2str (i))
			wt[][i] = wttemp[p]
		else
			wave wtemp = $("temp" + num2str (i))
			w[][i] = wtemp[p]
		endif
	endfor
	execute ("killwaves " + tempstr)
end
//function sort2D (windex, w)
//	wave w, windex
//	variable i, numrows = dimsize (w, 0), numcols = dimsize(w, 1)
//	string tempstr
//	
//	for (i = 0; i < numcols; i += 1)
//		make /o/n=(numrows) $("temp" + num2str (i))
//		wave wtemp = $("temp" + num2str (i))
//		wtemp = w[p][i]
//		if (i == 0)
//			tempstr = "temp0"
//		else
//			tempstr += ",temp" + num2str (i)
//		endif
//	endfor
//	execute ("sort " + nameofwave (windex) + " " + tempstr)	// do sort
//	for (i = 0; i < numcols; i += 1)	// recombine waves
//		wave wtemp = $("temp" + num2str (i))
//		w[][i] = wtemp[p]
//	endfor
//	execute ("killwaves " + tempstr)
//end
//function sort2Dindextext (windex, w)
//	wave w
//	wave /t windex
//	variable i, numrows = dimsize (w, 0), numcols = dimsize(w, 1)
//	string tempstr
//	
//	for (i = 0; i < numcols; i += 1)	// split columns up into individual waves
//		make /o/n=(numrows) $("temp" + num2str (i))
//		wave wtemp = $("temp" + num2str (i))
//		wtemp = w[p][i]
//		if (i == 0)
//			tempstr = "temp0"
//		else
//			tempstr += ",temp" + num2str (i)
//		endif
//	endfor
//	execute ("sort /a " + nameofwave (windex) + " " + tempstr)	// do sort
//	for (i = 0; i < numcols; i += 1)	// recombine waves
//		wave wtemp = $("temp" + num2str (i))
//		w[][i] = wtemp[p]
//	endfor
//	execute ("killwaves " + tempstr)
//end
//function sort2Dbothtext (windex, w)
//	wave/t windex,w
//	variable i, numrows = dimsize (w, 0), numcols = dimsize(w, 1)
//	string tempstr
//	
//	for (i = 0; i < numcols; i += 1)	// split columns up into individual waves
//		make /t/o/n=(numrows) $("temp" + num2str (i))
//		wave/t wtemp = $("temp" + num2str (i))
//		wtemp = w[p][i]
//		if (i == 0)
//			tempstr = "temp0"
//		else
//			tempstr += ",temp" + num2str (i)
//		endif
//	endfor
//	execute ("sort /a " + nameofwave (windex) + " " + tempstr)	// do sort
//	for (i = 0; i < numcols; i += 1)	// recombine waves
//		wave/t wtemp = $("temp" + num2str (i))
//		w[][i] = wtemp[p]
//	endfor
//	execute ("killwaves " + tempstr)
//end

function xwave2pnt (w, xval)
	// assuming w is a wave of x values (monotonic increasing),
	// returns point closest to xval
	wave w
	variable xval
	variable p1, p2, thisp
	
	if (numpnts (w) == 0)
		return -1
	endif
	p1 = 0
	p2 = numpnts (w) - 1
	do
		thisp = round ((p2 + p1) / 2)
		if (w[thisp] > xval)
			p2 = thisp
		else
			p1 = thisp
		endif
	while (p2 - p1 > 1)
	if (w[p2] - xval > xval - w[p1])
		return p1
	else
		return p2
	endif
end

function pmean (w, p1, p2)		// same as Igor mean function, but based on point values, rather than x values
	wave w
	variable p1, p2
	wavestats /q/r=[p1, p2] w
	return (v_avg)
end

function findwavestrval (w, val)
	wave /t w
	string val
	
	variable i = 0
	do
		if (cmpstr (w[i], val) == 0)
			return i
		endif
		i += 1
	while (i < numpnts (w))
	return (-1)
end

function avgxwaves (wavelist, wavexlist, destwave, prec)
	// average together waves that may or may not overlap x vals
	string wavelist, wavexlist, destwave
	variable prec
	string currwave, currwavex
	variable i = 0, j, wavepos
	
	make /o/n=0 $destwave, $(destwave + "_x"), $(destwave + "_count")
	wave wd = $destwave
	wave wdx = $(destwave + "_x")
	wave wdc = $(destwave + "_count")
	
	do															// go through list of waves
		printf ("*")
		currwave = stringfromlist (i, wavelist, ";")
		currwavex = stringfromlist (i, wavelist, ";")
		if ((!exists (currwave)) && (strlen (currwave) !=0))
			abort "Wave " + currwave + " doesn't exist."
		endif
		if ((!exists (currwavex)) && (strlen (currwavex) !=0))
			abort "Wave " + currwavex + " doesn't exist."
		endif
		if ((strlen (currwave) != 0) && (strlen (currwavex) != 0))	// is it a real wave?
			wave w = $currwave
			wave wx = $currwavex
			j = 0
			do													// go through x values
				printf (".")
				wavepos = findwaveval2 (wdx, wx[j], prec)		// is it already there?
				if (wavepos == -1)								// no, so add it
					wavepos = numpnts (wd)
					redimension /n=(wavepos + 1) wd, wdx, wdc
					wdx[wavepos] = wx[j]
					wd[wavepos] = w[j]
					wdc[wavepos] = 1
				else
					wd[wavepos] = (wd[wavepos] * wdc[wavepos] + w[j]) / (wdc[wavepos] + 1)
					wdc[wavepos] += 1
				endif
				j += 1
			while (j < numpnts (wx))
		endif
		i += 1
	while (strlen (currwave) != 0)
	sort wdx wdx, wd, wdc
	print "Done"
end


function avgmatches (basename, wstart, wend, destbase, matchvar, skips, includeSD)	// goes through a list of waves to average together similar ones
	string basename	// prefix for wavenames (e.g. "w")
	variable wstart,wend	// wave index numbers to go through (e.g. 12, 37)
	string destbase	// prefix for storing results (e.g. "pre" or "wash")
	string matchvar	// wave note variable to sort waves by (e.g. "PATTERN", or "stimInterval")
	wave skips		// wave of wave indexes to be skipped (because of bad noise, etc.)
	variable includeSD	// Boolean - 1 to calculate standard deviation, 0 if just average
	variable i
	string varval
	variable varpos, nomatch=0
	variable ILoadedIt

	make /o/n=0 $(destbase + "_count")
	make /o/n=0/t $(destbase + "_val")
	wave wcount = $(destbase + "_count")
	wave /t wval = $(destbase + "_val")
	
	for (i = wstart; i <= wend; i += 1)
		mafprogress ( (i - wstart) / (wend - wstart))
		if (findwaveval (skips, i) == -1)	// should we skip this one?
			if (!exists (basename + num2str (i)))
				ILoadedIt = 1
				LoadWave/q/P=home (basename + num2str (i) + ".bwav")
			else
				ILoadedIt = 0
			endif
			wave w = $(basename + num2str (i))
			if (strlen (matchvar) != 0)
				varval = stringbykey (matchvar, note(w))
				varpos = findwavestrval (wval, varval)						// has this value been found before?
			else
				varpos= 0
				if (numpnts (wcount) == 0)
					varpos = -1;varval = ""
				endif
			endif
			if ((strlen (varval) == 0) && (strlen (matchvar) != 0))		// doesn't have any value for matchvar
				nomatch += 1
			elseif (varpos == -1)											// not found, so allocate new one
				duplicate/o w $(destbase + "_" + varval)
				if (includeSD)
					duplicate /o w $(destbase + "_" + varval + "_SD")
					wave wvx2 = $(destbase + "_" + varval + "_SD")
					wvx2 *= w
				endif
				redimension /n=(numpnts (wcount) + 1) wcount, wval
				wcount[numpnts (wcount) - 1] = 1
				wval[numpnts (wcount) - 1] = varval
			else															// found, so update old one
				wave wvx = $(destbase + "_" + varval)
				wvx += w
				if (includeSD)
					wave wvx2 = $(destbase + "_" + varval + "_SD")
					wvx2 += w * w
				endif
				wcount[varpos] += 1
			endif
			if (ILoadedIt)
				killwaves w
			endif
		endif
	endfor
	for (i = 0; i < numpnts (wcount); i += 1)
		wave wvx = $(destbase + "_" + wval[i])
		if (includeSD)
			wave wvx2 = $(destbase + "_" + wval[i] + "_SD")
			wvx2 = sqrt ((wvx2 - wvx * wvx / wcount[i]) / (wcount[i] - 1))
		endif
		wvx /= wcount[i]
		print "     " + wval[i] + " (N = " + num2str (wcount[i]) + " / " + num2str (wend - wstart + 1) + ") --> " + destbase +"_" + wval[i]
	endfor
	if (nomatch > 0)
		print "     #No value# (N = " + num2str (nomatch) + " / " + num2str (wend - wstart + 1) + ")"
	endif
end

function stderr (w, pstart, pend)		// returns standard error over a region of a wave
	wave	w
	variable	pstart, pend
	wavestats /q/r=[pstart, pend] w
	return ((v_npnts < 2) ? 0 : (v_sdev / sqrt (v_npnts)))
end
function stdev (w, pstart, pend)
	wave w
	variable pstart, pend
	wavestats /q/r=[pstart, pend] w
	return (v_sdev)
end

function mafDFF (w1, w0, destname, basestart,  baseend, shutteroff, bkgnd, furalike, usefit)	// calculates deltaF/F
	wave w1, w0					// w1 = wave, w0 = baseline wave
	String destname				// where to store results
	Variable basestart,  baseend	// x values for measuring baseline (with shutter open)
	variable shutteroff				// time of shutter closing at end of trial
	variable bkgnd				// bkgnd = background fluorescence
	variable furalike				// 1 if dye is like fura (gets dimmer with Ca binding) or 0 if not (gets brighter with Ca)
	variable usefit				// 1 if fitting exponential to w0 to do deltaF/F calculation (if noise is a problem)
	
	Variable b1, b0				// baselines of wave, control (post shutter opening)
	variable z1, z0				// zero offsets of photodiode  (pre-shutter opening)
	variable coef				// temp coefficient for deltaF/F computation
	variable /g V_fitoptions=4
	
	if (usefit)
		CurveFit /q/w=0 exp w0(basestart, shutteroff)
	endif
	b1 = mean (w1, basestart, baseend)
	b0 = mean (w0, basestart, baseend)
	z1 = mean (w1, 0, .01)
	z0 = mean (w0, 0, .01)
	Duplicate /O w1, $destname
	wave DF = $destname
	z1 += bkgnd; z0 += bkgnd	// absorb background fluorescence into offset terms
	coef = -100 * (b0 - z0) / (b1 - z1)
	if (usefit)
		wave w_coef=w_coef
		DF = (w1 - z1) * coef / ((W_coef[0]+W_coef[1]*exp(-W_coef[2]*x)) - z0) + 100
	else
		DF = (w1- z1) * coef / (w0 - z0) + 100
	endif
	if (!furalike)
		df *= -1
	endif
	DF[0, x2pnt (df, basestart) - 1]=0
	if (shutteroff > baseend)
		DF[x2pnt (df, shutteroff) + 1, numpnts (DF) - 1] = 0
	endif
	SetScale d 0,0,"", DF
end

function decimaterange (w, destName, startpt, endpt, factor, renew)
	// decimates part of a wave, by clipping it out, decimating, and
	// reassembling -- doesn't allow slop at the edges
	// set renew to 1 the first time through and to 0 after there is an x-wave
	wave w
	string destName
	variable startpt, endpt, factor, renew
	variable numclip, offset
	
	if (mod (endpt - startpt + 1, factor) != 0)
		abort "Decimate aborted: Messy number of points specified."
	endif
	
	// first make x wave if necessary
	if ((!exists (NameofWave (w) + "_x")) || renew)
		duplicate /o w $(NameofWave (w) + "_x")
		wave wx = $(NameofWave (w) + "_x")
		wx = x
	else
		wave wx = $(NameofWave (w) + "_x")
	endif
	
	// now clip out parts to be decimated
	duplicate /o/r=[startpt, endpt] w, TempDecimateRange
	fdecimate (TempDecimateRange, "TempDecimateRange", factor)
	duplicate /o/r=[startpt, endpt] wx, tempdecimaterangex
	fdecimate (TempDecimateRangeX, "TempDecimateRangeX", factor)
	
	// now reassemble
	numclip = endpt - startpt + 1
	offset = numpnts (TempDecimateRange)
	duplicate /o w TempDecimateRange2
	duplicate /o wx TempDecimateRangeX2
	redimension /n=(numpnts (TempDecimateRange2) - (numclip - offset)) TempDecimateRange2
	redimension /n=(numpnts (TempDecimateRangeX2) - (numclip - offset)) TempDecimateRangeX2
	TempDecimateRange2[0, startpt - 1] = w[p]
	TempDecimateRangeX2[0, startpt - 1] = wx[p]
	TempDecimateRange2[startpt, startpt + offset - 1] = TempDecimateRange[p - startpt]
	TempDecimateRangeX2[startpt, startpt + offset - 1] = TempDecimateRangeX[p - startpt]
	TempDecimateRange2[startpt + offset, ] = w[p - offset + numclip]
	TempDecimateRangeX2[startpt + offset, ] = wx[p - offset + numclip]
	duplicate /o TempDecimateRange2, $destname
	duplicate /o TempDecimateRangeX2, $(destname + "_x")
	
	killwaves TempDecimateRange, TempDecimateRangeX
	killwaves TempDecimateRange2, TempDecimateRangeX2
End

function downsampleWaves (basename, wstart, wend, destname, factor)
	string basename, destname
	variable wstart, wend, factor
	variable i
	
	for (i = wstart; i <= wend; i += 1)
		wave w=$(basename + num2str (i))
		make /o/n=(numpnts (w)/factor) $(destname + num2str (i))
		wave wd=$(destname + num2str (i))
		copyscales w wd
		wd=w(x)
	endfor
end

function thistau ()	// returns the tau for the wave with the cursors on it on the top graph
	return posttau (csrwaveref(a), xcsr(a), xcsr(b))
end
function posttau (w, startx, endx)	// returns the tau of decay for a wave over the range specified
	wave w
	variable startx, endx
	variable tempoptions
	
	if (exists ("v_fitoptions") != 2)
		variable /g v_fitoptions = 0
	endif
	NVAR v_fitoptions
	tempoptions = v_fitoptions
	v_fitoptions = 4
	duplicate /o/r=(startx, endx) w temptau
	SetScale/P x 0,deltax (w),"", temptau
	CurveFit /Q/W=0 exp temptau
	v_fitoptions = tempoptions
	return (1/k2)
end
function avgtau (basename, wavenum, avgnum, startx, endx)
	string basename
	variable wavenum, avgnum, startx, endx
	variable j, avgminus, avgplus
	
	avgminus = wavenum - round ((avgnum - 1) / 2)
	avgplus = avgminus + avgnum - 1
	duplicate /o $(basename + num2str (avgminus)) wtemp
	for (j = avgminus + 1; j <= avgplus; j += 1)
		wave w=$(basename + num2str (j))
		wtemp += w
	endfor
	wtemp /= avgnum
	return (posttau (wtemp, startx, endx))
end

function EPSCpeak (w, startx, endx)	// returns the peak negative value over the range specified for a wave
	wave w
	variable startx, endx
	return (PSCPeak (w, startx, endx, -1))
end
function EPSCMaxpeak (w, startx, endx)	// returns the maximum value, negative or positive, over the range specified
	wave w
	variable startx, endx
	
	wavestats /q/r=(startx, endx) w
	return (-v_min > v_max ? v_min : v_max)
end
function PSCPeak (w, xstart, xend, pol)		// returns the maximum value over the range specified, depending on the polarity specified
	wave w
	variable xstart, xend, pol	
	wavestats /q/r=(xstart, xend) w
	return (pol == -1 ? v_min : v_max)
end
function PSCLatency (w, xstart, xend, pol)	// looks for max peak in derivative over interval specified
	wave w
	variable xstart, xend, pol
	duplicate /o/r=(xstart, xend) w wtemp
	differentiate wtemp
	wavestats /q wtemp
	return (pol == -1 ? v_minloc : v_maxloc)
end

function EPSCintegrate (w, startx, endx, basestart, baseend)	// returns the integral of a wave over the interval specified
	wave w
	variable startx, endx, basestart, baseend
	variable tempval
	
	duplicate /o/r=(startx, endx) w wtemp
	if (baseend != 0)
		tempval = mean (w, basestart, baseend)
		wtemp -= tempval
	endif
	integrate wtemp
	return (wtemp(endx))
end

function displayrange (basename, wstart, wend, yoffset)	// displays a bunch of waves
	string basename
	variable wstart, wend, yoffset
	variable i, j
	
	display $(basename + num2str (wstart))
	i = wstart + 1; j = 1
	do
		if (exists (basename + num2str (i)))
			appendtograph $(basename + num2str (i))
			modifygraph offset[j]={0, j * yoffset}
			j += 1
		endif
		i += 1
	while (i <= wend)
end

function displaymatches (basename, wstart, wend, yoffset, matchVar, matchVal)		// displays waves in the range specified, provided wavenote variable matchVar that matches matchVal
	string basename
	variable wstart, wend, yoffset
	string matchVar, matchVal
	variable i, j=0
	
	for (i = wstart; i <= wend; i += 1)
		if (cmpstr (stringbykey (matchvar, note ($(basename + num2str (i)))), matchval) == 0)
			if (j ==0)
				display $(basename + num2str (i))
			else
				appendtograph $(basename + num2str (i))
				modifygraph offset[j]={0, j * yoffset}
			endif
			j+= 1
		endif
	endfor
end


function displayrange2 (basename, wstart, wend, xstart, xend, yoffset)	// displays a bunch of waves, spacing waves by yoffset over the range xstart to xend
	string basename
	variable wstart, wend, xstart, xend, yoffset
	variable i = wstart, thisavg
	
	for (i = wstart; i <= wend; i += 1)
		if (i == wstart)
			display $(basename + num2str (i))
		else
			appendtograph $(basename + num2str (i))
		endif
		thisavg = mean ($(basename + num2str (i)), xstart, xend)
		modifygraph offset[i - wstart]={0, (i-wstart) * yoffset - thisavg}
	endfor
end

function displaythese (basename, indexwave, yoffset)		// displays waves with index values in indexwave
	string basename
	wave indexwave
	variable yoffset
	variable i = 0, thisavg
	
	for (i = 0; i < numpnts (indexwave); i += 1)
		if (i == 0)
			display $(basename + num2str (indexwave[i]))
		else
			appendtograph $(basename + num2str (indexwave[i]))
		endif
		modifygraph offset[i]={0, i * yoffset}
	endfor
end

function displayexceptthese (basename, wstart, wend, skipwave, yoffset)	// displays waves in range, unless their indices are in skipwave
	string basename
	variable wstart, wend
	wave skipwave
	variable yoffset
	variable i = wstart, thisavg, j = 0
	
	for (i = wstart; i <= wend; i += 1)
		if (findwaveval (skipwave, i) == -1)
			if (j == 0)
				display $(basename + num2str (i))
			else
				appendtograph $(basename + num2str (i))
			endif
			modifygraph offset[j]={0, j * yoffset}
			j += 1
		endif
	endfor
end
function displaymatchesexceptthese (basename, wstart, wend, yoffset, matchvar, matchVal, skipwave)	// displays matching waves in range, unless their indices are in skipwave
	string basename
	variable wstart, wend
	wave skipwave
	string matchVar, matchVal
	variable yoffset
	variable i = wstart, thisavg, j = 0
	
	for (i = wstart; i <= wend; i += 1)
		if (findwaveval (skipwave, i) == -1 && cmpstr (stringbykey (matchvar, note ($(basename + num2str (i)))), matchval) == 0)
			if (j == 0)
				display $(basename + num2str (i))
			else
				appendtograph $(basename + num2str (i))
			endif
			modifygraph offset[j]={0, j * yoffset}
			j += 1
		endif
	endfor
end


function findHalfHeight (w, displayit)		// finds the width of a pulse at half height, and optionally displays marker for where it was found
	wave w
	variable displayit
	variable halfval, start
	variable ravg, endpos, startpos
	
	wavestats /q w
	start = v_maxloc
	halfval = mean (w, start - .0005, start + .0005) / 2
	startpos = start
	do
		ravg = mean (w, startpos - .0005, startpos + .0005)
		if (ravg < halfval)
			break
		endif
		startpos -= deltax (w)
	while (startpos > 0)
	if (startpos <= 0)
		abort "Can't find start of pulse."
	endif
	
	endpos = start
	do
		ravg = mean (w, endpos -.005, endpos + .005)
		if (ravg < halfval)
			break
		endif
		endpos += deltax (w)
	while (endpos < rightx (w))
	if (endpos >= rightx (w))
		abort "Can't find end of pulse."
	endif
	
	if (displayit == 1)
		make /o/n=2 halfheight, halfheightx
		halfheight = halfval
		halfheightx = {startpos, endpos}
		getwindow kwTopWin, wavelist	// double check if it is already there
		WAVE /t w_wavelist=$("w_wavelist")
		if (findwavestrval (w_wavelist, "halfheight") == -1)
			appendtograph halfheight vs halfheightx
		endif
	endif
	return (endpos - startpos)
end
function findHalfHeightLim (w, xstart, xend, preavg, postavg, pol)		// returns the width at half-height for a pulse in a given range
	wave w
	variable xstart, xend, preavg, postavg, pol
	variable halfval, start
	variable ravg, endpos, startpos
	
	xstart = max (xstart, leftx (w))
	xend = min (xend, rightx (w))
	wavestats /q/r=(xstart, xend) w
	if (pol == -1)
		start = v_minloc
	else
		start = v_maxloc
	endif
	halfval = mean (w, start - preavg/2, start + preavg/2) / 2
	startpos = start
	do
		ravg = mean (w, startpos - preavg/2, startpos + preavg/2)
		if ((pol == -1 && ravg > halfval) || (pol == 1 && ravg < halfval))
			break
		endif
		startpos -= deltax (w)
	while (startpos >= xstart)
	if (startpos < xstart)
		print "Can't find start of pulse."
		return -1
	endif
	
	endpos = start
	do
		ravg = mean (w, endpos - postavg/2, endpos + postavg/2)
		if ((pol == -1 && ravg > halfval) || (pol == 1 && ravg < halfval))
			break
		endif
		endpos += deltax (w)
	while (endpos <= xend)
	if (endpos > xend)
		print "Can't find end of pulse."
		return -1
	endif
	return (endpos - startpos)
end

function mafprogress (currstatus)	// makes a progress bar
	variable currstatus
	
	if (currstatus >= 1)	// i.e. done
		dowindow /K ShowProgress
	else
		dowindow /f ShowProgress
		if (v_flag == 0)
			NewPanel /W=(288,226.25,561,278.75)/k=1 as "mafProgress"
			ValDisplay valdisp0,pos={0,17},size={272,18},title="Progress", limits={0,1,0},barmisc={0,1},value= #"k0"
			dowindow /c ShowProgress
		endif
		k0=currstatus
		controlupdate /w=ShowProgress valdisp0
//		doupdate
	endif
end


function GroupMean (m1, n1, m2, n2)	// given means and n's for two different groups, returns the mean for the two groups merged
	variable m1, n1, m2, n2
	if ((numtype (m1) != 2) && (numtype (m2) != 2))	//i.e. both legal numbers
		return (m1 * n1 + m2 * n2) / (n1 + n2)
	endif
	if (numtype (m1) != 2)
		return m1
	endif
	if (numtype (m2) != 2)
		return m2
	endif
	return (NaN)
end

function GroupSE (m1, se1, n1, m2, se2, n2)	// given means, std. errors and n's for two different groups, returns the standard error for the two groups merged
	variable m1, se1, n1, m2, se2, n2
	if ((numtype (m1) != 2) && (numtype (m2) != 2) && (numtype (se1) != 2) && (numtype (se2) != 2))
		return sqrt ((n1 * (n1 - 1) * se1^2 + n2 * (n2 - 1) * se2^2 + n1 * n2 * (m1 - m2)^2 / (n1 + n2)) / (n1 + n2) / (n1 + n2 - 1))
	endif
	if ((numtype (m1) != 2) && (numtype (se1) != 2))
		return se1
	endif
	if ((numtype (m2) != 2) && (numtype (se2) != 2))
		return se2
	endif
	return (NaN)
end

Function halfcircle(w,x) : FitFunc	// fit function for a half circle
	Wave w
	Variable x

	//CurveFitDialog/ These comments were created by the Curve Fitting dialog. Altering them will
	//CurveFitDialog/ make the function less convenient to work with in the Curve Fitting dialog.
	//CurveFitDialog/ Equation:
	//CurveFitDialog/ f(x) = sqrt (r^2 - (A * (x - x0))^2)
	//CurveFitDialog/ End of Equation
	//CurveFitDialog/ Independent Variables 1
	//CurveFitDialog/ x
	//CurveFitDialog/ Coefficients 3
	//CurveFitDialog/ w[0] = r
	//CurveFitDialog/ w[1] = x0
	//CurveFitDialog/ w[2] = A

	return (abs (w[2] * (x - w[1])) > abs (w[0]) ? 0 : sqrt (w[0]^2 - (w[2] * (x - w[1]))^2))
End


function killglitch (w, start, width)
	wave w
	variable start, width
//	w[start, start + width - 1] = (w[start-1] + w[start+width])/2
	w[start, start + width - 1] = w[start - 1] + (p - start + 1) / (width + 1) * (w[start + width] - w[start - 1])
end

function killthisglitch (width)	// kills a glitch in a wave that has a cursor on it
	variable width
	killglitch (csrwaveref (a), pcsr (a), width)
end

function EliminateGlitches (w)	// automatically deletes glitches from a wave
	// glitch must be more than 20 standard deviations outside noise,
	// and can't be riding on a transient greater than 3 standard devs
	wave w
	variable i = 1
	variable glitch = 0
	
	wavestats /q/r=(.1, .15) w
	do
		if ((abs (w[i + 1] - w[i - 1]) < v_sdev * 3) && (abs (w[i + 1] - w[i]) > 20 * v_sdev))
			glitch = 1
			w[i] = (w[i - 1] + w[i + 1]) / 2
		endif
		i += 1
	while (i < numpnts (w))
	return (glitch)
end
function AddGlitchButtons()
	SetVariable setvar0,pos={5,7},size={100,15},title="# points"
	SetVariable setvar0,limits={1,Inf,1},value= K0
	Button button0,pos={110,5},size={50,20},proc=KillGlitchButtonProc,title="Smooth"
	Button button1 proc=NaNGlitchProc,title="NaN"
	k0=0
	showinfo
EndMacro
Function KillGlitchButtonProc(ctrlName) : ButtonControl
	String ctrlName
	KillThisGlitch(k0)
End
Function NaNGlitchProc(ctrlName) : ButtonControl
	String ctrlName
	wave w=csrwaveref (a)
	w[pcsr (a), pcsr (a)+k0 - 1]=NaN
End

function KillTrainArtifacts (w, startx, ival, numstim, dtime)	// deletes train of stimulus artifacts in a wave
	wave w
	variable startx, ival, numstim, dtime
	variable i, tempval,dpoints
	
	if (dtime == 0)
		return 0
	endif
	dpoints = x2pnt (w, dtime)
	for (i = 0; i < numstim; i += 1)
		tempval = x2pnt (w, startx + ival * i)
		w[tempval, tempval + dpoints - 1] = nan 
	endfor
end
function KillArtifacts (w, stimTimes, stimtime)	// deletes stimulus artifacts with times given by wave stimTimes
	wave w, stimTimes
	variable stimtime
	variable stimpnts = x2pnt (w, stimtime)
	variable i, tempval
	
	if (stimpnts == 0)
		return 0
	endif
	for (i = 0; i < numpnts (stimTimes); i += 1)
		tempval = x2pnt (w, stimTimes[i])
		w[tempval, tempval + stimpnts - 1] = nan
	endfor
end